/*jshint esversion: 6 */

define(["lib/dev/config", "src/utils", "lodash", "immutable",
	"src/math/Mat3",
	"lib/tasks/dofs"],
function(config, utils, lodash, immutable,
	mat3,
	dofs) {
"use strict";

var exports = {};

var arrayPush = Array.prototype.push,
	dofTypeMerge = dofs.type.merge,
	dofTypePropagate = dofs.type.propagate,
	kFloatEps = Math.pow(2, -23), 
	kDofNotSet = dofs.type.kNotSet;

function getJointHandle (layer) {
	var parentPuppet = layer.parentPuppet,
		parentLayer = parentPuppet.getParentLayer(),
		handle = parentPuppet.getAttachToHandle(layer);

	if (!handle) return null;

	// use leaf proxy when layer attaches to origin
	if (handle === parentPuppet.getHandleTreeRoot()) {
		handle = parentLayer.getHandleJoint();
	}

	return handle;
}	
exports.getJointHandle = getJointHandle;


// FIXME: fix getParentLayer() so that it returns Layer not TrackItem
exports.getParentLayer = function (puppet) {
	var layer = puppet.getParentLayer(); // maybe layer

	// will work for real Layer and for TrackItem
	return layer.getSdkLayer().privateLayer;
};

// FIXME: fix when cleaning up retrieval of TrackItem layer.
// for example, test what happens when you change top-most puppet to warp with parent.
var getKeyPath = lodash.rest(function getKeyPath (layer, selectors) {

	var path = {
		layer,
		key : []
	};

	// note: parentPuppet not defined for layer-like track item
	while (layer.parentPuppet) {
		path.key.unshift("children", layer.getStageId());
		layer = layer.parentPuppet.getWarperLayer();
		path.layer = layer;
	}

	arrayPush.apply(path.key, selectors);

	utils.assert(path.layer === path.layer.getSdkLayer().privateLayer, "fix Layer retrieval for TrackItem Layer");

	return path;
});
exports.getKeyPath = getKeyPath;

exports.listInit = function (layer, generateItem) {
	return immutable.List().withMutations(function (l) {
		var tree = layer.getHandleTreeArray(),
			size = tree.size;

		l.setSize(size);
		while (--size >= 0) l.set(size, generateItem(tree.aHandle[size]));
	});
};

exports.listUpdate = function (layer, lom, updateItem) {
	return lom.withMutations(function (lom) {
		var tree = layer.getHandleTreeArray(),
			size = tree.size;

		while (--size >= 0) lom.update(size, updateItem(tree.aHandle[size]));
	});
};

// FIXME: precompute this matrix, it's static.
// Express handle anchor -- without handle motion -- frame relative to its parent
var matParent_Anchor = mat3();
function anchorFrameRelativeToParent(tree, handleRef) {
	var handle = tree.aHandle[handleRef];

	handle.getMatrixAtRestRelativeToParent(matParent_Anchor);
	if (handleRef > 0) {
		// FIXME: rename matLayerParent_Puppet -> matLayer_Source
		var matLayerParent_Puppet = tree.aMatLayerParent_Puppet[handleRef];
		if (matLayerParent_Puppet) mat3.multiply(matLayerParent_Puppet, matParent_Anchor, matParent_Anchor);
	} 

	return matParent_Anchor;
}

// Gather Handle frames (from LOM) expressed relative to Puppet frame.
// Optionally, supply mat_Puppet0 to express relative to another frame (i.e. Layer).
// Returns gathered frames as array of objects with matrix value and type.
// ALERT: This function can be a bottleneck. All changes should maintain efficiency.
function gatherHandleFrames (tree, lom, mat_Puppet0) {
	var matPuppet_Anchor = mat3(),
		length = tree.aHandle.length;

	var aFrames = new Array(length);

	// setup first iteration
	var handleRef = 0,
		parentRef = false,
		tParentFrame = {
			matrix : mat_Puppet0 || mat3.identity(),
			type : kDofNotSet
		};

	// walk to leaves accumulating matrices and types
	while (handleRef < length) {
		var mdof = lom.get(handleRef),
			type = mdof.get("type"),
			matAnchor_Handle = mdof.get("matrix");

		var matParent_Anchor = anchorFrameRelativeToParent(tree, handleRef);
		mat3.multiply(tParentFrame.matrix, matParent_Anchor, matPuppet_Anchor);
		var matrix = mat3.multiply(matPuppet_Anchor, matAnchor_Handle);

		// propagate type of all but the root handle which has the special handling
		if (parentRef && parentRef > 0)	{
			type = dofTypeMerge(dofTypePropagate(tParentFrame.type), type);
		}

		var anchor = mat3.from(matPuppet_Anchor);
		aFrames[handleRef] = { type, matrix, anchor };

		// setup next iteration
		handleRef += 1;
		parentRef = tree.aParentRef[handleRef];
		tParentFrame = aFrames[parentRef];
	}

	return aFrames;
}
    
function gatherHandleFramesNative (tree, lom, mat_Puppet0) {
	var matPuppet_Anchor = mat3(),
		length = tree.aHandle.length;

	var aFrames = new Array(length);

	// setup first iteration
	var handleRef = 0,
		parentRef = false,
		tParentFrame = {
			matrix : mat_Puppet0 || mat3.identity(),
			type : kDofNotSet
		};

	// walk to leaves accumulating matrices and types
	while (handleRef < length) {
		var mdof = lom.getDof(handleRef),
			type = mdof.type,
			matAnchor_Handle = mdof.matrix;

		var matParent_Anchor = anchorFrameRelativeToParent(tree, handleRef);
		mat3.multiply(tParentFrame.matrix, matParent_Anchor, matPuppet_Anchor);
		var matrix = mat3.multiply(matPuppet_Anchor, matAnchor_Handle);

		// propagate type of all but the root handle which has the special handling
		if (parentRef && parentRef > 0)	{
			type = dofTypeMerge(dofTypePropagate(tParentFrame.type), type);
		}

		var anchor = mat3.from(matPuppet_Anchor);
		aFrames[handleRef] = { type, matrix, anchor };

		// setup next iteration
		handleRef += 1;
		parentRef = tree.aParentRef[handleRef];
		tParentFrame = aFrames[parentRef];
	}

	return aFrames;
}
exports.gatherHandleFrames = config.MutableTree.enabled ? gatherHandleFramesNative : gatherHandleFrames;

// Scatter arrays relative to Puppet frame into LOM.
// Optionally, supply mat_Puppet0 for frames expressed relative to another frame (i.e. Layer).
// Returns updated LOM.
function scatterHandleFramesMutable (tree, aFrames, lom, mat_Puppet0) {
	var size = tree.size;

    var	matPuppet_Anchor = mat3();

    // walk down to leaves:
    // - accumulate previous matrices and types
    // - compare against provided
    // - update when different and preserve otherwise

    // setup first iteration
    var handleRef = 0,
        parentRef = false,
        tParentFrame = {
            matrix : mat_Puppet0 || mat3.identity(),
            type : kDofNotSet
        };

    while (handleRef < size){
        var matParent_Anchor = anchorFrameRelativeToParent(tree, handleRef);
        mat3.multiply(tParentFrame.matrix, matParent_Anchor, matPuppet_Anchor);

        var tFrame = aFrames[handleRef];

        //console.logToUser("in Matrix : " + tFrame.matrix);
        var isSingular = (Math.abs(matPuppet_Anchor[0]*matPuppet_Anchor[4] - matPuppet_Anchor[1]*matPuppet_Anchor[3]) < kFloatEps);
        if(isSingular)
            matPuppet_Anchor = mat3.identity();

        lom.setDof(handleRef, new dofs.Matrix({
            type : tFrame.type,
            matrix : mat3.multiply(matPuppet_Anchor.invert(), tFrame.matrix)
        }));

        // setup next iteration
        handleRef += 1;
        parentRef = tree.aParentRef[handleRef];
        tParentFrame = aFrames[parentRef];
    }
}
    
// Scatter arrays relative to Puppet frame into LOM.
// Optionally, supply mat_Puppet0 for frames expressed relative to another frame (i.e. Layer).
// Returns updated LOM.
function scatterHandleFrames (tree, aFrames, lomPrev, mat_Puppet0) {
	var size = tree.size;

	var lom = lomPrev.withMutations(function (lomPrev) {
		var	matPuppet_Anchor = mat3();

		// walk down to leaves:
		// - accumulate previous matrices and types
		// - compare against provided
		// - update when different and preserve otherwise

		// setup first iteration
		var handleRef = 0,
			parentRef = false,
			tParentFrame = {
				matrix : mat_Puppet0 || mat3.identity(),
				type : kDofNotSet
			};

		while (handleRef < size){
			var matParent_Anchor = anchorFrameRelativeToParent(tree, handleRef);
			mat3.multiply(tParentFrame.matrix, matParent_Anchor, matPuppet_Anchor);

			var tFrame = aFrames[handleRef];

			//console.logToUser("in Matrix : " + tFrame.matrix);
			var isSingular = (Math.abs(matPuppet_Anchor[0]*matPuppet_Anchor[4] - matPuppet_Anchor[1]*matPuppet_Anchor[3]) < kFloatEps);
			if(isSingular)
				matPuppet_Anchor = mat3.identity();

			lomPrev.set(handleRef, new dofs.Matrix({
				type : tFrame.type,
				matrix : mat3.multiply(matPuppet_Anchor.invert(), tFrame.matrix)
			}));

			// setup next iteration
			handleRef += 1;
			parentRef = tree.aParentRef[handleRef];
			tParentFrame = aFrames[parentRef];
		}
	});

	return lom;
}
exports.scatterHandleFrames = config.MutableTree.enabled ? scatterHandleFramesMutable : scatterHandleFrames;

function scatterLeafFrames (tree, aPuppetFrames, aMatLeaf, lomPrev) {
	var aLeafRef = tree.getLeafRefArray(),
		matPuppet_Anchor = mat3(),
		matAnchor_Puppet = mat3();

	var lom = lomPrev.withMutations(function (lom) {

		var index = aLeafRef.length;

		while (--index >= 0) {
			var refLeaf = aLeafRef[index],
				matPuppet_Leaf = aMatLeaf[index],
				frameParent = aPuppetFrames[tree.aParentRef[refLeaf]];

			var matParent_Anchor = anchorFrameRelativeToParent(tree, refLeaf);

			// the condition takes care of a tree with a single node: leaf without a parent
			if (frameParent) 
				mat3.multiply(frameParent.matrix, matParent_Anchor, matPuppet_Anchor);
			else 
				mat3.initWithArray(matParent_Anchor, matPuppet_Anchor);

			try {
				mat3.invert(matPuppet_Anchor, matAnchor_Puppet);
				lom.setIn([refLeaf, "matrix"], mat3.multiply(matPuppet_Anchor.invert(), matPuppet_Leaf));
			} catch (e) {
				lom.setIn([refLeaf, "matrix"], mat3.identity());
			}
		}
	});

	return lom;
}
exports.scatterLeafFrames = scatterLeafFrames;

    
function scatterLeafFramesMutable (tree, aPuppetFrames, aMatLeaf, lom) {
	var aLeafRef = tree.getLeafRefArray(),
		matPuppet_Anchor = mat3(),
		matAnchor_Puppet = mat3();

    var index = aLeafRef.length;

    while (--index >= 0) {
        var refLeaf = aLeafRef[index],
            matPuppet_Leaf = aMatLeaf[index],
            frameParent = aPuppetFrames[tree.aParentRef[refLeaf]];

        var matParent_Anchor = anchorFrameRelativeToParent(tree, refLeaf);

        // the condition takes care of a tree with a single node: leaf without a parent
        if (frameParent) 
            mat3.multiply(frameParent.matrix, matParent_Anchor, matPuppet_Anchor);
        else 
            mat3.initWithArray(matParent_Anchor, matPuppet_Anchor);

        // current thought is that we don't need to update timestamps here because we only need to track
        //  handle changes, not resulting warp-matrix changes; so we just make the change directly
        //  TODO: actually, setMatrix does touch now, so this may be inefficient
        try {
            mat3.invert(matPuppet_Anchor, matAnchor_Puppet);
            lom.setMatrix(refLeaf, mat3.multiply(matPuppet_Anchor.invert(), matPuppet_Leaf));
        } catch (e) {
            lom.setMatrix(refLeaf, mat3.identity());
        }
    }

	return lom;
}
exports.scatterLeafFramesMutable = scatterLeafFramesMutable;
    

/**
 * Test if the argument is a valid handle ref.
 * Note: Don't rely on simple falsey test because zero is falsey and we use it as a valid handle ref.
 */
function isValidRef (ref) {
	return typeof ref === "number" && ref >= 0;
}

function getPuppetFrame(tree, result0) {
	// FIXME: rename matLayerParent_Puppet -> matLayer_Source
	return mat3.initWithArray(tree.aMatLayerParent_Puppet[0], result0);
}
exports.getPuppetFrame = getPuppetFrame;

function getAnchorFrame (tree, lom, fromRef, toRef, result0) {
	var matParent_Anchor = anchorFrameRelativeToParent(tree, fromRef),
		matTo_From = mat3.initWithArray(matParent_Anchor, result0);


	var ref = tree.aParentRef[fromRef];
	while (isValidRef(ref) && ref >= toRef) {
		var matAnchor_Ref = lom.getIn([ref, "matrix"]);
		mat3.multiply(matAnchor_Ref, matTo_From, matTo_From);
		matParent_Anchor = anchorFrameRelativeToParent(tree, ref);
		mat3.multiply(matParent_Anchor, matTo_From, matTo_From);

		ref = tree.aParentRef[ref];
	}

	// null indicates Layer frame -- the parent of root handle	
	if (toRef === null) {
		mat3.multiply(getPuppetFrame(tree), matTo_From, matTo_From);
	}

	return matTo_From;
}
exports.getAnchorFrame = getAnchorFrame;

function getHandleFrame (tree, lom, fromRef, toRef, result0) {
	var matTo_Anchor = getAnchorFrame(tree, lom, fromRef, toRef),
		matAnchor_Handle = lom.getIn([fromRef, "matrix"]);

	return mat3.multiply(matTo_Anchor, matAnchor_Handle, result0);
}

exports.getHandleFrame = getHandleFrame;

/*
 * Calculates the matrix of a layer frame (layerOf) relative to its ancestor layer (layerTo).
 * NOTE: It traverses the entire state for the nested layer/puppet hierarchy.
 */
function getLayerFrame (tom, layerOf, layerTo, result0) {
	// currently supporting independent layers only
	utils.assert(!layerOf.getWarpWithParent() && !layerTo.getWarpWithParent());

	var path = getKeyPath(layerOf).key,
		result = result0 || mat3();

	function copyMatrix (src, dst) {
		return function (si, di) {
			dst[di] = src[si].matrix;
		};
	}

	var layer = layerOf;
	while (layer !== layerTo && layer && layer.parentPuppet) {
		const joint = getJointHandle(layer),
			parent = layer.parentPuppet.getWarperLayer();

		// remove last two entries for path to parent: ["children", id]
		path = path.slice(0, -2);

		const lom = tom.getIn(path).get("value"),
			tree = parent.getHandleTreeArray(),
			aHandleFrames = gatherHandleFrames(tree, lom);

		// TODO: Refactor with treeOfMatrix:warp()
		var matWarper_Joint;
		if (joint) {
			// Calculate sub-puppet position from attach-to handle.
			const refJoint = tree.getHandleRef(joint);
			matWarper_Joint = aHandleFrames[refJoint].matrix;
			// console.logToUser(`joint: ${matWarper_Joint}`);
		} else {
			// Calculate sub-puppet position from attach-to coordinates.
			var	aLeafRef = tree.getLeafRefArray(),
				nLeaf = aLeafRef.length,
				aMatLeafFrames = new Array(nLeaf);

			aLeafRef.forEach(copyMatrix(aHandleFrames, aMatLeafFrames));
			matWarper_Joint = parent.getWarpAtAttachment(aMatLeafFrames, layer);
			// console.logToUser(`attach to: ${matWarper_Joint}`);
		}
		layer.filterJointFrame(matWarper_Joint, matWarper_Joint);
		// console.logToUser(`filtered: ${matWarper_Joint}`);

		const matParent_Joint = mat3.multiply(tree.aMatLayerParent_Puppet[0], matWarper_Joint);
		// console.logToUser(`matParent_Joint: ${matParent_Joint}`);
		mat3.multiply(matParent_Joint, result, result);
		layer = parent;
	}

	utils.assert(layer === layerTo, "convertLayerFrameRelativeToAncestor(): expected to find ancestor, but none found.");

	// console.logToUser(`matTo_Of: ${result}`);

	return result;
}
exports.getLayerFrame = getLayerFrame;

/*
 * Convert frame matrix (matTrack_Frame) expressed relative to track layer into 
 * equivalent matrix (matLayer_Frame) expressed relative to the provided layer.
 * NOTE: It uses uncommitted state (tomNext) to enable sequential (root-to-leaf) update of nested layer/puppet objects.
 */
function convertTrackLayerFrame (layer, layerTrack, matTrack_Frame, result0) {
	var tom = layerTrack.tomNext,
		matTrack_Layer = getLayerFrame(tom, layer, layerTrack);

	// invariant: matTrack_Frame = matTrack_Layer matLayer_Frame
	return mat3.multiply(mat3.invert(matTrack_Layer), matTrack_Frame, result0);
}
exports.convertTrackLayerFrame = convertTrackLayerFrame;

return exports;

}); // end define
